/*
 *  This file is part of MagiCapture.
 *  Copyright (C) 1999 Arthur Jerijian
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * panel.c: Routines for managing the widgets on the main panel
 */

#include <stdio.h>

#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <magick/api.h>
#include <gdk_magick.h>

#include "common.h"
#include "capture.h"
#include "options.h"
#include "help.h"
#include "save.h"
#include "stringtable.h"
#include "widget.h"
#include "util.h"
#include "panel.h"

/* Global variables */

static GtkWidget *panel_window = NULL;

/* Screen capture parameters */
static CaptureType cap_type = DEFAULT_TYPE;
static guint cap_delay = DEFAULT_DELAY;

/* ImageMagick image data structures */
static Image *magick_image = NULL;
static ImageInfo *magick_image_info = NULL;

/* Options */
static OptionData option_data;

/*********************************************************************
 *
 * Main panel window callback functions
 */

gint panel_delete_event (GtkWidget *widget, GdkEvent *event, gpointer data)
{
    return (FALSE);
}

void panel_destroy_event (GtkWidget *widget, gpointer data)
{
    gtk_main_quit ();
}

/*********************************************************************
 *
 * panel_destroy_image: Destroy the ImageMagick data structures and
 * free the structures' allocated memory.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void panel_destroy_image ()
{
    if (magick_image != NULL)
    {
        DestroyImage (magick_image);
        magick_image = NULL;
    }
    if (magick_image_info != NULL)
    {
        DestroyImageInfo (magick_image_info);
        magick_image_info = NULL;
    }
}

/*********************************************************************
 *
 * panel_store_options: Retrieve new options from the options dialog
 * box and store them for later use by the rest of the program.
 *
 * Parameters:
 *   new_option_data - options retrieved by the options dialog box
 *
 * Return value:
 *   None.
 */

void panel_store_options (OptionData *new_option_data)
{
    option_data.preview = new_option_data -> preview;
    option_data.hide_panel = new_option_data -> hide_panel;
    option_data.dither = new_option_data -> dither;
}

/*********************************************************************
 *
 * panel_init: Create the main panel widget which includes
 * all buttons, labels, and text boxes belonging to the main panel.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void panel_init ()
{
    GtkWidget *box;
    GSList *group;

    panel_window = NULL;
    
    /* Set default screen capture parameters. */
    
    cap_type = DEFAULT_TYPE;
    cap_delay = DEFAULT_DELAY;

    /* Initialize all data structure pointers to NULL. */
    
    magick_image = NULL;
    magick_image_info = NULL;

    /* Create the main panel window. */
    
    panel_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

    gtk_signal_connect
    (
        GTK_OBJECT (panel_window),
        "delete_event",
        GTK_SIGNAL_FUNC (panel_delete_event),
        NULL
    );
    gtk_signal_connect
    (
        GTK_OBJECT (panel_window),
        "destroy",
        GTK_SIGNAL_FUNC (panel_destroy_event),
        NULL
    );

    gtk_window_set_title (GTK_WINDOW (panel_window), TITLE_PANEL);
    gtk_container_set_border_width (GTK_CONTAINER (panel_window), 10);

    /* Create the box that will hold all the buttons and entry fields. */
    
    box = gtk_vbox_new (FALSE, 0);

    /*
     * Add radio buttons for capture mode: single window,
     * single window with decorations, and full screen.
     */
    
    add_label (box, LABEL_CAPTURE);
    
    group = add_radio
    (
        box,
        NULL,
        NULL,
        GTK_SIGNAL_FUNC (panel_single_func),
        TRUE,
        LABEL_SINGLE_WINDOW
    );
    group = add_radio
    (
        box,
        NULL,
        group,
        GTK_SIGNAL_FUNC (panel_single_decor_func),
        FALSE,
        LABEL_SINGLE_WINDOW_DECOR
    );
    group = add_radio
    (
        box,
        NULL,
        group,
        GTK_SIGNAL_FUNC (panel_full_func),
        FALSE,
        LABEL_FULL_SCREEN
    );

    /* Add delay parameter widgets. */
    
    add_label (box, LABEL_DELAY);
    add_delay (box, GTK_SIGNAL_FUNC (panel_delay_func));

    add_separator (box);

    /* Add the capture and preview buttons. */
    
    add_button
    (
        box,
        NULL,
        GTK_SIGNAL_FUNC (panel_capture_func),
        LABEL_CAPTURE_DO
    );
    add_button
    (
        box,
        NULL,
        GTK_SIGNAL_FUNC (panel_preview_func),
        LABEL_PREVIEW
    );
    
    add_separator (box);

    /*
     * Add buttons for configuring options, saving the capture to
     * a file, and quitting the program.
     */
    
    add_button
    (
        box,
        NULL,
        GTK_SIGNAL_FUNC (panel_options_func),
        LABEL_OPTIONS
    );
    add_button
    (
        box,
        NULL,
        GTK_SIGNAL_FUNC (panel_saveas_func),
        LABEL_SAVEAS
    );
    add_button
    (
        box,
        NULL,
        GTK_SIGNAL_FUNC (panel_quit_func),
        LABEL_QUIT
    );

    add_separator (box);

    /* Add a button for displaying the software version and copyright. */

    add_button
    (
        box,
        NULL,
        GTK_SIGNAL_FUNC (panel_about_func),
        LABEL_ABOUT
    );

    /* Add the box to the main panel window. */
    
    gtk_container_add (GTK_CONTAINER (panel_window), box);

    /* Show all objects. */
    
    gtk_widget_show (box);
    gtk_widget_show (panel_window);
}

/*********************************************************************
 *
 * panel_cleanup: Free all memory allocated to main panel data
 * structures. This function must be called after the GTK main event
 * loop.
 *
 * Parameters:
 *   None.
 *
 * Return value:
 *   None.
 */

void panel_cleanup ()
{
    panel_destroy_image ();
}

/*********************************************************************
 *
 * panel_single_func: GTK callback function called when the user
 * selects the "Single Window" radio button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_single_func (GtkWidget *widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON (widget) -> active)
    {
        debug_print (LABEL_SINGLE_WINDOW);
        cap_type = SINGLE_WINDOW;
    }
}

/*********************************************************************
 *
 * panel_single_decor_func: GTK callback function called when the user
 * selects the "Single Window with Decorations" radio button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_single_decor_func (GtkWidget *widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON (widget) -> active)
    {
        debug_print (LABEL_SINGLE_WINDOW_DECOR);
        cap_type = SINGLE_WINDOW_DECOR;
    }
}

/*********************************************************************
 *
 * panel_full_func: GTK callback function called when the user
 * selects the "Full Screen" radio button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_full_func (GtkWidget *widget, gpointer data)
{
    if (GTK_TOGGLE_BUTTON (widget) -> active)
    {
        debug_print (LABEL_FULL_SCREEN);
        cap_type = FULL_SCREEN;
    }
}

/*********************************************************************
 *
 * panel_delay_func: GTK callback function called when the user
 * modifies the contents of the delay parameter text box.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_delay_func (GtkWidget *widget, gpointer data)
{
#ifdef MAGICAPTURE_DEBUG    
    GString *delay_debug_buffer;
#endif /* MAGICAPTURE_DEBUG */

    cap_delay = gtk_spin_button_get_value_as_int
    (
        GTK_SPIN_BUTTON (widget)
    );

#ifdef MAGICAPTURE_DEBUG    
    delay_debug_buffer = g_string_new (NULL);
    g_string_sprintf
    (
        delay_debug_buffer,
        "%s%d%s",
        LABEL_DELAY,
        cap_delay,
        LABEL_SECONDS
    );
    debug_print (delay_debug_buffer -> str);
    g_string_free (delay_debug_buffer, TRUE);
#endif /* MAGICAPTURE_DEBUG */
}

/*********************************************************************
 *
 * panel_capture_func: GTK callback function called when the user
 * clicks on the "Capture!" button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_capture_func (GtkWidget *widget, gpointer data)
{
    gint cap_status;
    
    debug_print (LABEL_CAPTURE_DO);

    /* Hide the main panel as applicable. */
    
    if (option_data.hide_panel)
    {
        gtk_widget_hide (panel_window);
    }
    else
    {
        /* Change the panel's cursor to an hourglass. */
        gdk_window_set_cursor
        (
            widget -> parent -> window,
            gdk_cursor_new (GDK_WATCH)
        );
    }
    
    /* Take a screen capture. */
    
    panel_destroy_image ();
    cap_status = capture_do
    (
        &magick_image,
        &magick_image_info,
        cap_type,
        cap_delay
    );
    if (!cap_status)
    {
        return;
    }

    /* Create a capture preview window. */
    
    gdk_magick_set_dither (option_data.dither);
    cap_status = capture_preview (magick_image);
    if (!cap_status)
    {
        return;
    }

    /* Redisplay the main panel. */
    
    gtk_widget_show (panel_window);

    /* If applicable, preview the capture window. */
    
    if (option_data.preview)
    {
        capture_show ();
    }

    /* Restore the main panel's cursor to the left-arrow pointer. */
    
    gdk_window_set_cursor
    (
        widget -> parent -> window,
        gdk_cursor_new (GDK_LEFT_PTR)
    );
}

/*********************************************************************
 *
 * panel_preview_func: GTK callback function called when the user
 * clicks on the "Preview" button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_preview_func (GtkWidget *widget, gpointer data)
{
    debug_print (LABEL_PREVIEW);

    /* Display the capture preview window. */
    
    capture_show ();
}

/*********************************************************************
 *
 * panel_options_func: GTK callback function called when the user
 * clicks on the "Options..." button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_options_func (GtkWidget *widget, gpointer data)
{
    debug_print (LABEL_OPTIONS);

    /* Prompt the user to set options. */
    
    options_do (&option_data);
}

/*********************************************************************
 *
 * panel_saveas_func: GTK callback function called when the user
 * clicks on the "Save As..." button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_saveas_func (GtkWidget *widget, gpointer data)
{
    debug_print (LABEL_SAVEAS);

    /* Open a dialog box to save the screen capture. */
    
    save_do (magick_image, magick_image_info);
}

/*********************************************************************
 *
 * panel_quit_func: GTK callback function called when the user
 * clicks on the "Quit" button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_quit_func (GtkWidget *widget, gpointer data)
{
    debug_print (LABEL_QUIT);

    /* Exit the GTK main event loop. */
    
    gtk_main_quit ();
}

/*********************************************************************
 *
 * panel_about_func: GTK callback function called when the user
 * clicks on the "About" button.
 *
 * Parameters:
 *   widget - GTK widget connected to the callback function
 *   data - callback function data
 *
 * Return value:
 *   None.
 */

void panel_about_func (GtkWidget *widget, gpointer data)
{
    debug_print (LABEL_ABOUT);

    /* Display an about dialog box. */
    
    help_about ();
}

